/**
 * Copyright 2014 JogAmp Community. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of
 *       conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list
 *       of conditions and the following disclaimer in the documentation and/or other materials
 *       provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY JogAmp Community ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JogAmp Community OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are those of the
 * authors and should not be interpreted as representing official policies, either expressed
 * or implied, of JogAmp Community.
 */
package com.jogamp.plugin.applet;

import java.util.Locale;

import com.jogamp.plugin.ui.NativeWindowDownstream;
import com.jogamp.plugin.ui.NativeWindowUpstream;

/**
 * Implemented by user.
 * <a name="lifecycle"><h5>Applet3 Lifecycle</h5></a>
 * <p>
 * <ul>
 *   <li>{@link #createNativeWindow(Applet3Context, NativeWindowUpstream)}</li>
 *   <li>{@link #init(Applet3Context)}</li>
 *   <li>{@link #start()}</li>
 *   <li>{@link #stop()}</li>
 *   <li>{@link #destroy()}</li>
 * </ul>
 * </p>
 */
public interface Applet3 {

    /**
     * Returns applet information or <code>null</code>.
     * <p>
     * Implementation may utilize this method to
     * return information about the author, version, and copyright of the applet.
     * </p>
     */
    String getAppletInfo();

    /**
     * Returns a custom locale of the applet or <code>null</code>.
     */
    Locale getLocale();

    /**
     * Returns a description of parameters as used by this applet, or <code>null</code>.
     * <p>
     * The returned string array is arranged to
     * contain the <code>name</code>, the <code>type</code>, and a <code>description</code>.
     * Example:
     * <pre>
     * String pinfo[][] = {
     *   {"gl_profile",       "GLProfile String", "The GLProfile to be used, i.e. GL2ES2"},
     *   {"gl_swap_interval", "0-4",              "The swap interval for vertical sync, i.e. 0 or 1"},
     *   {"gl_debug",         "boolean",          "Enable JOGL debugging"}
     * };
     * </pre>
     */
    String[][] getParameterInfo();

    /**
     * Implementation creates a native child window, allowing to be controlled by the plugin.
     * <p>
     * The applet's child window is destroyed by the plugin after it has called {@link #destroy()}.
     * </p>
     * <p>
     * Note that the returned {@link NativeWindowDownstream} instance's {@link NativeWindowDownstream#getParent()}
     * must match the given <code>parent</code> instance, otherwise the applet will be aborted.
     * </p>
     * <p>
     * See <a href="#lifecycle">Applet Lifecycle</a>.
     * </p>
     * @param context the {@link Applet3Context}
     * @param parent the parent {@link NativeWindowUpstream}, reflecting the plugin's native applet window.
     * @return {@link NativeWindowDownstream} users native child window.
     */
    NativeWindowDownstream createNativeWindow(Applet3Context context, NativeWindowUpstream parent);

    /**
     * Initialize the applet. Implementation may allocate resources.
     * <p>
     * See <a href="#lifecycle">Applet Lifecycle</a>.
     * </p>
     *
     * @see #createNativeWindow(Applet3Context, NativeWindowUpstream)
     * @see #destroy()
     * @see #start()
     * @see #stop()
     */
    void init(Applet3Context context);

    /**
     * Start the applet.
     * <p>
     * See <a href="#lifecycle">Applet Lifecycle</a>.
     * </p>
     * @see #createNativeWindow(Applet3Context, NativeWindowUpstream)
     * @see #destroy()
     * @see #start()
     * @see #stop()
     */
    void start();

    /**
     * Stop the applet.
     * <p>
     * See <a href="#lifecycle">Applet Lifecycle</a>.
     * </p>
     * @see #createNativeWindow(Applet3Context, NativeWindowUpstream)
     * @see #destroy()
     * @see #start()
     * @see #stop()
     */
    void stop();

    /**
     * Destroy the applet and all it's resources. Implementation shall release resources as allocated via {@link #init(Applet3Context)}.
     * <p>
     * Note that the applet's child window {@link #createNativeWindow(Applet3Context, NativeWindowUpstream) created by this implementation}
     * is still valid and may be destroyed here.
     * </p>
     * <p>
     * See <a href="#lifecycle">Applet Lifecycle</a>.
     * </p>
     * @see #createNativeWindow(Applet3Context, NativeWindowUpstream)
     * @see #destroy()
     * @see #start()
     * @see #stop()
     */
    void destroy();
}
