#ifdef WIN32
# include <windows.h>
#endif

#include <GL/gl.h>
#undef GL_GLEXT_PROTOTYPES

#define GLH_EXT_SINGLE_FILE
#define GLH_NVEB_USING_NVPARSE

//#define LOCAL_DATA_FILES

#include <glh_nveb.h>
#include <glh_obs.h>
#include <glh_glut.h>

#include "CA_Water.hpp"

NVEB_EFFECT_NAME("OpenGL Water");
NVEB_EFFECT_VERSION("1.0");
NVEB_EFFECT_LOCATION("Effects\\Procedural");
NVEB_EFFECT_ABOUT(0, NULL, "OpenGL Water Simulation", "NVEffectsExplained.htm#OpengGLWater");
NVEB_EFFECT_ABOUT(1, "Date", "July 2001", NULL);
NVEB_EFFECT_ABOUT(2, NULL, "NVIDIA Developer Relations", "http://www.nvidia.com/Developer");


using namespace glh;

// Global variables

bool b[256];								// Toggle state for all the keys (characters)

CA_Water        *pWater;

int             iXResolution;               // window dimensions
int             iYResolution;
bool            bLeftButtonDown = false;
int             iMousePosX      = 0;
int             iMousePosY      = 0;

glut_perspective_reshaper reshaper;
glut_simple_mouse_interactor camera;

// glut callbacks
void Display();
void Key(unsigned char k, int x, int y);
void Menu( int v );
void Reshape(int w, int h);
void Idle();
void Mouse(int button, int state, int x, int y);
void Motion(int x, int y);

// other functions
void Initialize();
void MakeGlutWindowCurrent();

//.----------------------------------------------------------------------------.
//|   Function   : main                                                        |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
int main(int argc, char **argv)
{
    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA );
    glutCreateWindow("Procedural Texture Waves");

    b['w'] = false;
    b['s'] = true;
    b[' '] = true;
    b['l'] = true;
    b['o'] = true;
 
    Initialize();

    glut_helpers_initialize();
    camera.configure_buttons(1);
	camera.set_camera_mode(false);
    camera.dolly.dolly[2] = -2;
    //camera.pan.pan[1] = 1;

    glut_add_interactor(& reshaper);
	glut_add_interactor(& camera);

    glutCreateMenu( Menu );
    glutAddMenuEntry( "Wireframe toggle [w]",					    'w' );
    glutAddMenuEntry( "Draw in water / move quad toggle[d]",        'd' );
    glutAddMenuEntry( "Animation toggle[ ]",				        ' ' );
    glutAddMenuEntry( "Rotating Logo toggle [r]",				    'o' );
	glutAddMenuEntry( "Single step (when paused) [n]",   		    'n' );
    glutAddMenuEntry( "Slow / full-speed animation toggle [s]",     's' );
    glutAddMenuEntry( "Border wrapping toggle [b]",		            'b' );
    glutAddMenuEntry( "Reset velocity [r]",					        'r' );
    glutAddMenuEntry( "Blur distance: decrease [c]",                'c' );
    glutAddMenuEntry( "Blur distance: increase [v]",                'v' );
    glutAddMenuEntry( "Bump scale: decrease [-]",                   '-' );
    glutAddMenuEntry( "Bump scale: increase [=]",                   '=' );
    glutAddMenuEntry( "Droplet frequency: decrease [,]",            ',' );
    glutAddMenuEntry( "Droplet frequency: increase [.]",            '.' );
    glutAddMenuEntry( "Logo: display in water [l]",                 'l' );
    glutAddMenuEntry( "Display: Reflective Liquid Surface [1]",     '1' );
    glutAddMenuEntry( "Display: Height Map [2]", 		            '2' );
    glutAddMenuEntry( "Display: Force Map [3]", 		            '3' );
    glutAddMenuEntry( "Display: Normal Map [4]", 		            '4' );
    glutAddMenuEntry( "Display: Force, Velocity, Height and Normal Maps [5]",'5' );
    
    glutAttachMenu( GLUT_RIGHT_BUTTON );

    glutKeyboardFunc(Key);
    glutIdleFunc(Idle);
    glutDisplayFunc(Display);
    glutMouseFunc(Mouse);
    glutMotionFunc(Motion);
    glutMainLoop();
    return 0;
}


//.----------------------------------------------------------------------------.
//|   Function   : Initialize                                                  |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Initialize()
{
    if (!glh_init_extensions("\
         GL_ARB_multitexture \
         GL_NV_vertex_program \
         GL_NV_texture_shader \
         GL_NV_register_combiners \
         WGL_ARB_pbuffer \
         WGL_ARB_pixel_format \
        "))
    {
        printf("Demo requires the following extension(s): %s\n", glh_get_unsupported_extensions());
        exit(0);
    }

    glClearColor(0, 0.2f, 0.5f, 0);
    glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    gluPerspective(90, 1, .1, 10);
    
    pWater = new CA_Water;

    
#ifdef LOCAL_DATA_FILES
    pWater->Initialize("nvfixed.tga", "nvspin.tga", "droplet.tga", "CloudyHills_%s.tga");
#else
    pWater->Initialize( "../../data/images/nvfixed.tga", 
                        "../../data/images/nvspin.tga", 
                        "../../data/images/droplet.tga", 
                        "../../data/cubemaps/CloudyHills_%s.tga");
#endif

    MakeGlutWindowCurrent();
}


//.----------------------------------------------------------------------------.
//|   Function   : Menu                                                        |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Menu( int v )
{
    static float rBlurIncrement         = 0.01f;
    static float rBumpIncrement         = 0.01f;
    static float rFrequencyIncrement    = 0.1f;

    b[v] = !b[v];

    switch (v)
    {
    case 27:
    case 'q':
        exit(0);
    case 'w':
        pWater->EnableWireframe(b['w']);
        break;
    case 'd':
        if (b['d'])
        {
            glutMouseFunc(glh::glut_mouse_function);
            glutMotionFunc(glh::glut_motion_function);
        }
        else
        {
            glutMouseFunc(Mouse);
            glutMotionFunc(Motion);
        }
        break;
    case ' ':
        pWater->EnableAnimation(b[' ']);
        break;
    case 'b':
        pWater->EnableBorderWrapping(b['b']);
        break;
    case 'n':
        pWater->SingleStep();
        break;
    case 's':
        pWater->EnableSlowAnimation(b['s']);
        break;
    case '1':
        pWater->SetRenderMode(CA_Water::CA_FULLSCREEN_REFLECT);
        break;
    case '2':
        pWater->SetRenderMode(CA_Water::CA_FULLSCREEN_HEIGHT);
        break;
    case '3':
        pWater->SetRenderMode(CA_Water::CA_FULLSCREEN_FORCE);
        break;
    case '4':
        pWater->SetRenderMode(CA_Water::CA_FULLSCREEN_NORMALMAP);
        break;
    case '5':
        pWater->SetRenderMode(CA_Water::CA_TILED_THREE_WINDOWS);
        break;
    case 'r':
        pWater->Reset();
        break;    
    case 'i':
        gluPerspective(90, 1, .01, 10);
        break;
    case 'c':
        {
            float rDist = pWater->GetBlurDistance();
            if (rDist > rBlurIncrement)
                pWater->SetBlurDistance(rDist - rBlurIncrement);
            break;
        }
    case 'v':
        {
            float rDist = pWater->GetBlurDistance();
            if (rDist < 1)
                pWater->SetBlurDistance(rDist + rBlurIncrement);
            break;
        }
    case '-':
        {
            float rScale = pWater->GetBumpScale();
            if (rScale > -1)
                pWater->SetBumpScale(rScale - rBumpIncrement);
            break;
        }
    case '=':
        {
            float rScale = pWater->GetBumpScale();
            if (rScale < 1)
                pWater->SetBumpScale(rScale + rBumpIncrement);
            break;
        }
    case 'l':
        pWater->EnableBoundaryApplication(b['l']);
        break;
    case 'o':
        pWater->EnableSpinningLogo(b['o']);
        break;
    case '.':
        {
            float rFrequency = pWater->GetBumpScale();
            if (rFrequency < 1)
                pWater->SetDropFrequency(rFrequency + rFrequencyIncrement);
            break;
        }
    case ',':
        {
            float rFrequency = pWater->GetBumpScale();
            if (rFrequency > 0)
                pWater->SetDropFrequency(rFrequency - rFrequencyIncrement);
            break;
        }
    default:
        break;
    }

    glutPostRedisplay();
}


//.----------------------------------------------------------------------------.
//|   Function   : Key                                                         |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Key(unsigned char k, int x, int y)
{
    Menu((int)k);
}


//.----------------------------------------------------------------------------.
//|   Function   : Mouse                                                       |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Mouse(int button, int state, int x, int y)
{
    if (GLUT_LEFT_BUTTON == button && GLUT_DOWN == state )
    {
        bLeftButtonDown = true;
        iMousePosX = x;
        iMousePosY = y;
    }
    else if (GLUT_LEFT_BUTTON == button)
    {
        bLeftButtonDown = false;
    }
}


//.----------------------------------------------------------------------------.
//|   Function   : Motion                                                      |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Motion(int x, int y)
{
    iMousePosX = x;
    iMousePosY = y;
}

//.----------------------------------------------------------------------------.
//|   Function   : MakeGlutWindowCurrent                                       |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void MakeGlutWindowCurrent()
{
    static int siGlutWinId = glutGetWindow();
    glutSetWindow( siGlutWinId );
}


//.----------------------------------------------------------------------------.
//|   Function   : Idle                                                        |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Idle()
{
    glutPostRedisplay();
}

//.----------------------------------------------------------------------------.
//|   Function   : Display                                                     |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void Display()
{
    // draw in the liquid
    if (bLeftButtonDown)
    {   
        CA_Water::Droplet droplet = { 2 * (iMousePosX / (float) glutGet(GLUT_WINDOW_WIDTH) - 0.5f),
                                      -2 * (iMousePosY / (float) glutGet(GLUT_WINDOW_HEIGHT) - 0.5f),
                                      0.08f };
        pWater->AddDroplet(droplet);
    }

    pWater->Tick();

    MakeGlutWindowCurrent();

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
	
    camera.apply_transform();

    matrix4f orot;
	camera.trackball.r.get_value(orot);
    //printf("%f\n", object.pan.pan.v[0]);
    
    pWater->Display(orot);
   
    glutSwapBuffers();
}
