#ifndef __CA_WATER_HPP__
#define __CA_WATER_HPP__

#include <vector>
#include <glh_linear.h>

using namespace glh;

// forward declaration so we don't have to include the file here
class PBuffer;

class CA_Water
{
public:
	CA_Water();
	~CA_Water();

    void    Initialize(char *pInitialMapFilename, char *pSpinFilename, char *pDropletFilename, char *pCubemapFilename);

    void    Tick();
    void    Display(const matrix4f& matRotation);
    
    void    SingleStep()                               { _bSingleStep  = true;         }
    void    EnableAnimation(bool bEnable)              { _bAnimate     = bEnable;      }
    void    EnableSlowAnimation(bool bEnable)          { _bSlow        = bEnable;      }
    void    Reset()                                    { _bReset       = true;         }
   
    enum RenderMode
    {
            CA_FULLSCREEN_REFLECT,
            CA_FULLSCREEN_FORCE,
            CA_FULLSCREEN_HEIGHT,
            CA_FULLSCREEN_NORMALMAP,
            CA_TILED_THREE_WINDOWS,
            CA_DO_NOT_RENDER
    };

    void    SetRenderMode(RenderMode eMode)            { _eRenderMode  = eMode;        }
    
    void    EnableWireframe(bool bEnable)              { _bWireframe   = bEnable;      }
    void    EnableBorderWrapping(bool bEnable)         { _bWrap        = bEnable;      }
    
    void    EnableBoundaryApplication(bool bEnable)    { _bApplyInteriorBoundaries = bEnable;  }
    void    EnableSpinningLogo(bool bEnable)           { _bSpinLogo    = bEnable;      }

    void    SetBlurDistance(float rDistance)           { _rBlurDist    = rDistance; _UpdateBlurVertOffset();}
    float   GetBlurDistance() const                    { return _rBlurDist;            }

    void    SetBumpScale(float rScale)                 { _rBumpScale   = rScale;       }
    float   GetBumpScale() const                       { return _rBumpScale;           }

    void    SetDropFrequency(float rFrequency)         { _rDropletFrequency = rFrequency;   }
    float   GetDropFrequency() const                   { return _rDropletFrequency;         }

    struct Droplet
    {
        float rX;
        float rY;
        float rScale;
    };

    void AddDroplet(const Droplet &drop);

    unsigned int GetHeightTextureID() const            { return _pDynamicTextureIDs[CA_TEXTURE_HEIGHT_TARGET]; }
    unsigned int GetNormalMapTextureID() const         { return _pDynamicTextureIDs[CA_TEXTURE_NORMAL_MAP];    }

private: // methods
    

    void _DoSingleTimeStep();
    void _CreateNormalMap();
    void _LoadTextures(char *pInitialMapFilename, char *pSpinFilename, char *pDropletFilename, char *pCubeMapFilename);
    void _CreateAndWriteUVOffsets(unsigned int width, unsigned int height);
    void _UpdateBlurVertOffset();
    void _DrawInteriorBoundaryObjects();
    void _DrawDroplets();

private: // data

    enum StaticTextureNames
    {
        CA_TEXTURE_INITIAL_MAP,
        CA_TEXTURE_SPIN,
        CA_TEXTURE_DROPLET,
        CA_TEXTURE_CUBEMAP,
        CA_NUM_STATIC_TEXTURES
    };

    enum DynamicTextureNames
    {
        CA_TEXTURE_FORCE_INTERMEDIATE,
        CA_TEXTURE_FORCE_TARGET,
        CA_TEXTURE_VELOCITY_SOURCE,
        CA_TEXTURE_VELOCITY_TARGET,
        CA_TEXTURE_HEIGHT_SOURCE,
        CA_TEXTURE_HEIGHT_TARGET,
        CA_TEXTURE_NORMAL_MAP,
        CA_NUM_DYNAMIC_TEXTURES
    };
    
    enum ListNames
    {
        CA_REGCOMBINER_EQ_WEIGHT_COMBINE,
        CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_1,
        CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_2,
        CA_REGCOMBINER_APPLY_FORCE,
        CA_REGCOMBINER_APPLY_VELOCITY,
        CA_REGCOMBINER_CREATE_NORMAL_MAP,
        CA_TEXTURE_SHADER_REFLECT,
        CA_DRAW_SCREEN_QUAD,
        CA_NUM_LISTS
    };

    unsigned int    _pInitialMapDimensions[2];  // the dimensions of the initial map.
    
    unsigned int    _pStaticTextureIDs[CA_NUM_STATIC_TEXTURES];     
    unsigned int    _pDynamicTextureIDs[CA_NUM_DYNAMIC_TEXTURES];
    
    unsigned int    _iTexHeightInput;           // current input height texture ID.
    unsigned int    _iTexHeightOutput;          // current output height texture ID.
    unsigned int    _iTexVelocityInput;         // current input velocity texture ID.
    unsigned int    _iTexVelocityOutput;        // current output velocity texture ID.
    unsigned int    _iTexForceStepOne;          // intermediate force computation result texture ID.
    unsigned int    _iTexForceOutput;           // current output force texture ID.

    //unsigned int    _iCurrentSourceID;          // the current "input" texture to an update step.
    
    unsigned int    _pDisplayListIDs[CA_NUM_LISTS];
    
    unsigned int    _iVertexProgramID;          // one vertex shader is used to choose the texcoord offset

    unsigned int    _iFlipState;                // used to flip target texture configurations.

    bool            _bWrap;                     // CA can either wrap its borders, or clamp (clamp by default)  
    bool            _bReset;                    // are we resetting this frame? (user hit reset).
    bool            _bSingleStep;               // animation step on keypress.
    bool            _bAnimate;                  // continuous animation.
    bool            _bSlow;                     // run slow.
    bool            _bWireframe;                // render in wireframe mode
    bool            _bApplyInteriorBoundaries;  // enable / disable "boundary" image drawing.
    bool            _bSpinLogo;                 // draw spinning logo.
    bool            _bCreateNormalMap;          // enable / disable normal map creation.

    float           _rPerTexelWidth;            // width of a texel (percentage of texture)
    float           _rPerTexelHeight;           // height of a texel

    float           _rBlurDist;                 // distance over which to blur.
    float           _rBlurIncrement;            // amount to change blur distance.

    float           _rNormalSTScale;            // scale of normals in normal map.
    float           _rBumpScale;                // scale of bumps in water.

    float           _rDropletFrequency;         // frequency at which droplets are drawn in water...

    unsigned int    _iSlowDelay;                // amount to delay when running slow.
    unsigned int    _iSkipInterval;             // frames to skip simulation.

    PBuffer         *_pPixelBuffer;             // pbuffer for rendering.

    std::vector<Droplet> _droplets;             // array of droplets

    RenderMode      _eRenderMode; 

    enum ConstantMemoryLocations
    {
        CV_WORLDVIEWPROJ_0 = 0,
        CV_WORLDVIEWPROJ_1 = 1,
        CV_WORLDVIEWPROJ_2 = 2,
        CV_WORLDVIEWPROJ_3 = 3,

        CV_UV_OFFSET_TO_USE = 4,

        CV_UV_T0_NO_OFFSET   = 8,
        CV_UV_T0_TYPE1       = 9,
        CV_UV_T0_TYPE2      = 10,
        CV_UV_T0_TYPE3      = 11,
        CV_UV_T0_TYPE4      = 12,

        CV_UV_T1_NO_OFFSET  = 13,
        CV_UV_T1_TYPE1      = 14,
        CV_UV_T1_TYPE2      = 15,
        CV_UV_T1_TYPE3      = 16,
        CV_UV_T1_TYPE4      = 17,

        CV_UV_T2_NO_OFFSET  = 18,
        CV_UV_T2_TYPE1      = 19,
        CV_UV_T2_TYPE2      = 20,
        CV_UV_T2_TYPE3      = 21,
        CV_UV_T2_TYPE4      = 22,

        CV_UV_T3_NO_OFFSET  = 23,
        CV_UV_T3_TYPE1      = 24,
        CV_UV_T3_TYPE2      = 25,
        CV_UV_T3_TYPE3      = 26,
        CV_UV_T3_TYPE4      = 27,

        CV_CONSTS_1         = 28
    };
    
};


#endif //__CA_WATER_HPP__